﻿// Khởi tạo kết nối SignalR tới hub cập nhật tiến trình ký số
const connection = new signalR.HubConnectionBuilder()
    .withUrl("/progressHub")
    .build();

const progressBarIds = ["tai_du_lieu", "ky_so", "gui_file", "tong_the"];


const getSelectedIds = () => {
    const table = $('#filesTable').DataTable();
    const selectedIds = [];

    table.rows().every(function () {
        const row = this.node();
        const checkbox = $(row).find('.row-checkbox');
        if (checkbox.is(':checked')) {
            const data = this.data(); // đây là row object
            if (data?.id) selectedIds.push(data.id); // hoặc data.Id tuỳ vào key bạn dùng
        }
    });

    return selectedIds;
};


/**
 * Trả về tổng số hồ sơ đang được chọn để ký.
 */
let totalDossierCount = 0;

let successCount = 0;
let failureCount = 0;
let stepProgress = {
    tai_du_lieu: 0,
    ky_so: 0,
    gui_file: 0
};

/**
 * Cập nhật trạng thái progress bar cho từng bước hoặc tổng thể.
 * Nếu hoàn thành, hiển thị trạng thái hoàn thành.
 * @param {string} id - Tên bước tiến trình hoặc 'tong_the'
 * @param {number} value - Số lượng đã xử lý hoặc phần trăm hoàn thành
 * @param {number} total - Tổng số lượng cần xử lý
 */
function updateProgressBar(id, value, total) {
    const bar = document.getElementById("progressBar_" + id);
    if (!bar) return;

    if (["tai_du_lieu", "ky_so", "gui_file"].includes(id)) {
        const percent = total > 0 ? (value / total * 100) : 0;
        bar.style.width = percent + "%";
        bar.textContent = `${value}/${total}`;

        // Cập nhật số lượng đã xử lý
        stepProgress[id] = value;

        // Cập nhật số công văn đã hoàn thành nếu tất cả bước đã xong
        const minCompleted = Math.min(stepProgress.tai_du_lieu, stepProgress.ky_so, stepProgress.gui_file);
        successCount = minCompleted;
    } else if (id === "tong_the") {
        bar.style.width = value + "%";
        bar.textContent = value + "%";
    }
}

/**
 * Chuyển UI sang trạng thái hoàn thành tiến trình ký.
 * Ẩn nút dừng, hiện nút đóng và hiển thị thông báo hoàn thành.
 * @param {string} message - Thông báo hoàn thành
 */
function toggleCompleteState(successMessage, errorMessage) {
    document.getElementById("btnStop")?.classList.add("d-none");
    document.getElementById("btnClose")?.classList.remove("d-none");

    const successMessageDiv = document.getElementById("successMessage");
    if (successMessageDiv) {
        successMessageDiv.textContent = successMessage;
        successMessageDiv.classList.remove("d-none");
    }
    const errorMessageDiv = document.getElementById("errorMessage");
    if (errorMessageDiv) {
        errorMessageDiv.textContent = errorMessage;
        errorMessageDiv.classList.remove("d-none");
    }
}
let processCanceled = false;
// Lắng nghe sự kiện cập nhật tiến trình từ server qua SignalR và cập nhật UI
connection.on("UpdateProgress", (processName, value, total) => {
    updateProgressBar(processName, value, total);
});

connection.on("Error", (errorMessage) => {
    showAlertModal(errorMessage, "danger");
});

connection.on("Complete", (successCount, errorCount) => {
    var errMsg = '';
    if (errorCount) {
        errMsg = `Không thể xử lý ${errorCount}/${totalDossierCount} công văn. Vui lòng kiểm tra lại.`;
    }
    toggleCompleteState(`Hoàn thành ký ${successCount}/${totalDossierCount} công văn`, errMsg);
});

// Bắt đầu kết nối SignalR
connection.start().catch(err => console.error(err.toString()));


/**
 * Xử lý sự kiện khi nhấn nút "Bắt đầu ký".
 * Kiểm tra hồ sơ được chọn, reset UI, hiển thị modal tiến trình và gửi request bắt đầu ký.
 */
let selectedSlotId = null;
let selectedCodes = [];


let allIdsToSign = [];

document.getElementById("btnSign")?.addEventListener("click", () => {
    const btn = document.getElementById("btnSign");
    disableButtonTemporarily(btn);
    const selected = getSelectedIds();
    totalDossierCount = selected.length;
    handleSign(selected, false);
});

document.getElementById("btnSignAll")?.addEventListener("click", async () => {
    const btn = document.getElementById("btnSignAll");
    disableButtonTemporarily(btn);
    try {
        const form = document.getElementById("filterForm");
        const formData = new FormData(form);

        const urlEncoded = new URLSearchParams();
        formData.forEach((value, key) => {
            urlEncoded.append(key, value);
        });

        const res = await fetch("/Home/GetAllFileToSignIds", {
            method: "POST",
            headers: {
                "Content-Type": "application/x-www-form-urlencoded"
            },
            body: urlEncoded.toString()
        });

        if (!res.ok) throw new Error(await res.text());

        allIdsToSign = await res.json();
        totalDossierCount = allIdsToSign.length;

        if (allIdsToSign.length === 0) {
            showAlertModal("Không có công văn nào đủ điều kiện để ký.", "danger");
            return;
        }

        // Gán số lượng vào modal
        document.getElementById("signAllCount").innerText = totalDossierCount;

        // Hiển thị modal xác nhận
        const modal = bootstrap.Modal.getOrCreateInstance(document.getElementById("confirmSignAllModal"));
        modal.show();
    } catch (err) {
        showAlertModal("Không thể lấy danh sách công văn để ký tất cả.", "danger");
        console.error(err);
    }
});

document.getElementById("confirmSignAllBtn")?.addEventListener("click", () => {
    const btn = document.getElementById("confirmSignAllBtn");
    disableButtonTemporarily(btn);
    const modal = bootstrap.Modal.getInstance(document.getElementById("confirmSignAllModal"));
    modal.hide();

    if (allIdsToSign.length > 0) {
        handleSign(allIdsToSign, true);
    }
});


async function handleSign(ids, isSignAll) {
    if (!ids || ids.length === 0) {
        showAlertModal("Vui lòng chọn ít nhất một công văn để thực hiện ký.", "danger");
        return;
    }

    selectedCodes = ids;
    resetUI();

    if (!isSignAll) {
        const hasPermission = await checkPermission(selectedCodes);
        if (!hasPermission) return;
    }

    try {
        const slots = await getSlots();
        document.getElementById("lbSlotSelect").innerText = "Chọn thiết bị: ";

        if (slots.length === 0) {
            showAlertModal("Không phát hiện thiết bị ký số. Vui lòng kết nối thiết bị để tiếp tục.", "danger");
            return;
        }

        if (slots.length === 1) {
            selectedSlotId = slots[0].id;
            const usbInfo = `Chữ ký số: ${slots[0].userCertName} (${slots[0].vendor} - ${slots[0].serialNumber})`;
            showPinModalSingle(usbInfo);
        } else {
            selectedSlotId = slots[0].id;
            populateSlotSelect(slots);
            $('#slotModal').modal('show');

            document.getElementById("slotSelect").onchange = (e) => {
                selectedSlotId = e.target.value;
            };
            document.getElementById("slotSelect").click();
            document.getElementById("lbSlotSelect").innerText = "Chọn thiết bị (" + slots.length + "): ";
        }
    } catch (error) {
        showAlertModal(error.message, "danger");
        console.error(error);
    }
}
function showPinModalSingle(usbInfo) {
    document.getElementById("usbInfoSingle").textContent = usbInfo || "Thông tin USB không xác định";

    document.getElementById("pinInputSingle").value = "";
    const errorDiv = document.getElementById("pinErrorSingle");
    errorDiv.classList.add("d-none");
    errorDiv.textContent = "";

    const modal = bootstrap.Modal.getOrCreateInstance(document.getElementById("pinModal"));
    modal.show();
}

function checkPermission(selectedCodes) {
    return new Promise((resolve) => {
        $.ajax({
            url: "/Home/CheckPermissionByFileIds",
            type: "POST",
            contentType: "application/json",
            data: JSON.stringify(selectedCodes),
            success: function () {
                resolve(true); // Cho phép tiếp tục
            },
            error: function (xhr) {
                if (xhr.responseJSON?.message) {
                    showAlertModal(xhr.responseJSON.message, "danger");
                } else {
                    showAlertModal("Lỗi không xác định.", "danger");
                }
                resolve(false); // Dừng xử lý
            }
        });
    });
}



// Xác nhận chọn USB
document.getElementById("slotSelect")?.addEventListener("change", () => {
    const select = document.getElementById("slotSelect");
    selectedSlotId = select.value;
});


// Xác nhận mã PIN
async function handlePinSubmit(pinInputId, errorDivId) {
    const btnSubmitPin = document.getElementById("btnSubmitPin");
    const btnSubmitPinSingle = document.getElementById("btnSubmitPinSingle");

    btnSubmitPin.disabled = true;
    btnSubmitPinSingle.disabled = true;

    document.getElementById(pinInputId).value = document.getElementById(pinInputId).value.trim(); // Optional: trim
    const pin = document.getElementById(pinInputId).value;
    const errorDiv = document.getElementById(errorDivId);
    errorDiv.classList.add("d-none");

    await verifyPin(selectedSlotId, pin)
        .then(() => {
            $('#pinModal').modal('hide');
            $('#slotModal').modal('hide');
            startProgress();
        })
        .catch(err => {
            errorDiv.classList.remove("d-none");
            errorDiv.textContent = err.message || "Mã PIN không đúng hoặc thiết bị ký số không thể xác thực. Vui lòng kiểm tra và thử lại.";
            document.getElementById(pinInputId).value = ""; // ✅ Reset input sau khi thất bại
        });

    btnSubmitPin.disabled = false;
    btnSubmitPinSingle.disabled = false;
}

document.getElementById("btnSubmitPin")?.addEventListener("click", async () => {
    await handlePinSubmit("pinInput", "pinError");
});

document.getElementById("btnSubmitPinSingle")?.addEventListener("click", async () => {

    await handlePinSubmit("pinInputSingle", "pinErrorSingle");
});


async function getSlots() {
    const res = await fetch("/Home/GetSlots", {
        headers: {
            "X-Requested-With": "XMLHttpRequest"
        }
    });
    if (!res.ok) {
        let errMessage = "Không thể lấy danh sách chữ ký số."
        const errorBody = await res.json();

        if (errorBody?.message) {
            errMessage = errorBody.message;
        }

        throw new Error(errMessage);
    }
    return res.json();
}


async function verifyPin(slotId, pin) {
    const res = await fetch("/Home/VerifyPin", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ SlotId: slotId, Pin: pin })
    });
    if (!res.ok) {
        const message = await res.text();
        throw new Error(message);
    }
}

function populateSlotSelect(slots) {
    const select = document.getElementById("slotSelect");
    select.innerHTML = "";
    slots.forEach(slot => {
        const option = document.createElement("option");
        option.value = slot.id;
        option.textContent = `${slot.userCertName} (${slot.vendor} - ${slot.serialNumber})`;
        select.appendChild(option);
    });
}

function startProgress() {
    fetch("/Home/StartProgress", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(selectedCodes)
    })
        .then(async response => {
            console.log("Failed to start progress:", response);
            if (!response.ok) {
                const error = await response.json().catch(() => ({ message: "Unknown error" }));
                throw error;
            }

            const modal = bootstrap.Modal.getOrCreateInstance(document.getElementById("progressModal"));
            modal.show();
        })
        .catch(err => {
            // err giờ có { code, message } từ backend
            if (err.code === "INVALID_USB") {
                showAlertModal(err.message, "danger");
            } else {
                showAlertModal(err.message || "Có lỗi xảy ra khi bắt đầu tiến trình ký.", "danger");
            }
            console.error(err);
        });
}

function resetUI() {
    successCount = 0;
    failureCount = 0;
    stepProgress = { tai_du_lieu: 0, ky_so: 0, gui_file: 0 };
    document.getElementById("btnStop")?.classList.remove("d-none");
    document.getElementById("btnClose")?.classList.add("d-none");

    const successMessageDiv = document.getElementById("successMessage");
    const errorMessageDiv = document.getElementById("errorMessage");
    if (successMessageDiv) {
        successMessageDiv.classList.add("d-none");
        successMessageDiv.textContent = "";
    }
    if (errorMessageDiv) {
        errorMessageDiv.classList.add("d-none");
        errorMessageDiv.textContent = "";
    }

    progressBarIds.forEach(id => {
        const bar = document.getElementById("progressBar_" + id);
        if (bar) {
            bar.style.width = "0%";
            bar.innerText = id === "tong_the" ? "0%" : "0/0";
        }
    });
}



/**
 * Xử lý sự kiện khi nhấn nút "Dừng tiến trình".
 * Hiển thị modal xác nhận hủy tiến trình ký.
 */
document.getElementById("btnStop")?.addEventListener("click", () => {
    const modal = bootstrap.Modal.getOrCreateInstance(document.getElementById("confirmCancelModal"));
    modal.show();
});

/**
 * Xử lý sự kiện khi xác nhận hủy tiến trình ký.
 * Gửi request dừng tiến trình, cập nhật UI và thông báo số lượng đã hoàn thành.
 */
document.getElementById("btnConfirmCancel")?.addEventListener("click", async () => {
    const btn = document.getElementById("btnConfirmCancel");
    disableButtonTemporarily(btn);
    var errMsg = '';
    if (failureCount) {
        errMsg = `Không thể xử lý ${failureCount}/${totalDossierCount} công văn. Vui lòng kiểm tra lại.`;
    }
    await stopProgress();
    toggleCompleteState(`Đã hoàn thành ${successCount} / ${totalDossierCount} công văn trước khi hủy.`, errMsg);
});


async function stopProgress() {
    try {
        const res = await fetch("/Home/StopProgress", { method: "POST" });
        const data = await res.json();
        successCount = data.successCount || successCount;
        failureCount = data.failureCount || failureCount;

        const modal = bootstrap.Modal.getInstance(document.getElementById("confirmCancelModal"));
        modal.hide();
    } catch (err) {
        console.error("Error stopping progress:", err);
    }
}


const disableButtonTemporarily = (btn, ms = 1500) => {
    btn.disabled = true;
    setTimeout(() => btn.disabled = false, ms);
};




/**
 * Hiển thị modal thông báo với nội dung và kiểu cảnh báo tương ứng.
 * @param {string} message - Nội dung thông báo
 * @param {string} type - Loại thông báo: info, success, warning, danger
 */
function showAlertModal(message, type = 'info') {
    const modalHeader = document.getElementById("customModalHeader");
    const modalBody = document.getElementById("customModalBody");
    const modalCloseBtn = document.getElementById("customModalBtnClose");

    // Reset style
    modalHeader.className = 'modal-header text-white';
    modalCloseBtn.className = 'btn';

    // Áp dụng kiểu theo loại
    switch (type) {
        case 'error':
        case 'danger':
            modalHeader.classList.add('bg-danger');
            modalCloseBtn.classList.add('btn-danger');
            break;
        case 'warning':
            modalHeader.classList.add('bg-warning');
            modalHeader.classList.add('text-dark');
            modalCloseBtn.classList.add('btn-warning');
            break;
        case 'success':
            modalHeader.classList.add('bg-success');
            modalCloseBtn.classList.add('btn-success');
            break;
        case 'info':
        default:
            modalHeader.classList.add('bg-info');
            modalCloseBtn.classList.add('btn-info');
            break;
    }

    modalBody.textContent = message;

    const modal = bootstrap.Modal.getOrCreateInstance(document.getElementById("customModalAlert"));
    modal.show();
}
